import tkinter as tk
from tkinter import ttk
import math
import matplotlib
import numpy as np
from matplotlib.backends.backend_tkagg import FigureCanvasTkAgg
import matplotlib.pyplot as plt
import re

matplotlib.use("TkAgg")

# ----------------- HDGL Machine -----------------
class HDGLMachine:
    def __init__(self):
        # Upper field primitives
        self.upper_field = {
            "phi": 1.6180339887,
            "phi_power_phi": 2.6180339887,
        }
        # Analog dimensions / braids
        self.analog_dims = {
            "P3": 4.2360679775,
            "P4": 6.8541019662,
        }
        # Lower field primitives
        self.lower_field = {
            "inv_phi": 0.6180339887,
            "inv_P3": 0.2360679775,
        }
        self.void = 0.0
        self.current_state = self.void
        self.dimension = 1.0  # 2D double-helix by default
        self.recursion_active = False
        self.seed = 1.0  # calculator result applied here

    # ---- HDGL Bootstrap Arithmetic ----
    def hdgl_add(self, a, b):
        # sum using φ as base primitive
        return a + b + self.upper_field["phi"]

    def hdgl_sub(self, a, b):
        # subtract using contraction primitive
        return a - b + self.lower_field["inv_phi"]

    def hdgl_mul(self, a, b):
        # multiply using Fibonacci braid P3
        return a * b + self.analog_dims["P3"]

    def hdgl_div(self, a, b):
        # divide using lower contraction primitive
        if b == 0:
            return float("inf")
        return a / b + self.lower_field["inv_P3"]

    def evaluate_expression(self, expr: str):
        """
        Parse a basic expression string and evaluate using HDGL operations.
        Supports + - * / with correct bootstrap precedence.
        """
        # tokenize numbers and operators
        tokens = re.findall(r"\d+\.\d+|\d+|[+\-*/]", expr)
        if not tokens:
            return 0.0

        # convert numbers
        def parse_num(tok): return float(tok) if re.match(r"\d", tok) else tok
        tokens = [parse_num(t) for t in tokens]

        # operator precedence: *, /
        i = 0
        while i < len(tokens):
            if tokens[i] == "*":
                res = self.hdgl_mul(tokens[i - 1], tokens[i + 1])
                tokens[i - 1:i + 2] = [res]
                i -= 1
            elif tokens[i] == "/":
                res = self.hdgl_div(tokens[i - 1], tokens[i + 1])
                tokens[i - 1:i + 2] = [res]
                i -= 1
            else:
                i += 1

        # operator precedence: +, -
        i = 0
        while i < len(tokens):
            if tokens[i] == "+":
                res = self.hdgl_add(tokens[i - 1], tokens[i + 1])
                tokens[i - 1:i + 2] = [res]
                i -= 1
            elif tokens[i] == "-":
                res = self.hdgl_sub(tokens[i - 1], tokens[i + 1])
                tokens[i - 1:i + 2] = [res]
                i -= 1
            else:
                i += 1

        return tokens[0]

    # ---- Machine dynamics ----
    def toggle_recursion(self):
        self.recursion_active = not self.recursion_active

    def compute_harmonic_state(self, t):
        state = self.void
        # simple harmonic example using φ
        state += math.sin(t * self.upper_field["phi"]) * self.seed
        return state

    def step(self, t):
        if self.recursion_active:
            self.current_state = self.compute_harmonic_state(t) * 2  # recursion doubles
        else:
            self.current_state = self.compute_harmonic_state(t)
        return self.current_state

    def kick(self, value: float):
        # inject calculator result
        self.seed = value if value != 0 else 1.0

# ----------------- GUI -----------------
class HDGLCalculatorApp:
    def __init__(self, root):
        self.root = root
        self.root.title("HDGL Machine Calculator")

        self.machine = HDGLMachine()
        self.expression = ""

        # Display
        self.display = tk.Entry(root, font=("Consolas", 18), bd=5, relief="sunken", justify="right")
        self.display.pack(fill="x", padx=10, pady=5)

        # Calculator buttons
        btns_frame = ttk.Frame(root)
        btns_frame.pack(padx=10, pady=5)

        buttons = [
            "7", "8", "9", "/",
            "4", "5", "6", "*",
            "1", "2", "3", "-",
            "0", ".", "=", "+"
        ]
        for i, text in enumerate(buttons):
            b = ttk.Button(btns_frame, text=text, command=lambda t=text: self.on_button(t))
            b.grid(row=i // 4, column=i % 4, sticky="nsew", padx=2, pady=2)

        for i in range(4):
            btns_frame.columnconfigure(i, weight=1)
        for i in range(4):
            btns_frame.rowconfigure(i, weight=1)

        # Control buttons
        ctrl_frame = ttk.Frame(root)
        ctrl_frame.pack(padx=10, pady=5, fill="x")
        ttk.Button(ctrl_frame, text="C", command=self.clear).pack(side="left", expand=True, fill="x")
        ttk.Button(ctrl_frame, text="Recursion", command=self.toggle_recursion).pack(side="left", expand=True, fill="x")

        # Waveform Plot
        self.fig, self.ax = plt.subplots(figsize=(6, 3))
        self.canvas = FigureCanvasTkAgg(self.fig, master=root)
        self.canvas.get_tk_widget().pack(fill="both", expand=True)
        self.line, = self.ax.plot([], [], lw=2)
        self.ax.set_xlim(0, 10)
        self.ax.set_ylim(-5, 5)
        self.t_vals = np.linspace(0, 10, 500)
        self.update_plot()

    def on_button(self, char):
        if char == "=":
            try:
                result = self.machine.evaluate_expression(self.expression)
                self.display.delete(0, tk.END)
                self.display.insert(0, str(result))
                self.expression = str(result)
                self.machine.kick(result)  # update waveform with bootstrap result
            except Exception:
                self.display.delete(0, tk.END)
                self.display.insert(0, "Error")
                self.expression = ""
        else:
            self.expression += str(char)
            self.display.delete(0, tk.END)
            self.display.insert(0, self.expression)

    def clear(self):
        self.expression = ""
        self.display.delete(0, tk.END)

    def toggle_recursion(self):
        self.machine.toggle_recursion()

    def update_plot(self):
        y_vals = [self.machine.step(t) for t in self.t_vals]
        self.line.set_data(self.t_vals, y_vals)
        self.ax.set_ylim(min(y_vals) - 1, max(y_vals) + 1)
        self.canvas.draw()
        self.root.after(200, self.update_plot)

# ----------------- Run -----------------
if __name__ == "__main__":
    root = tk.Tk()
    app = HDGLCalculatorApp(root)
    root.mainloop()
